<?php

namespace Spatie\Mailcoach\Livewire\Mails;

use Filament\Actions\Action;
use Filament\Actions\BulkAction;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Spatie\Mailcoach\Domain\Audience\Enums\SuppressionReason;
use Spatie\Mailcoach\Domain\Audience\Models\Suppression;
use Spatie\Mailcoach\Domain\Shared\Traits\UsesMailcoachModels;
use Spatie\Mailcoach\Livewire\TableComponent;

class SuppressionListComponent extends TableComponent
{
    use InteractsWithActions;
    use UsesMailcoachModels;

    public function table(Table $table): Table
    {
        return parent::table($table)
            ->query(self::getSuppressionClass()::query())
            ->columns([
                TextColumn::make('email')
                    ->label(__mc('Email'))
                    ->sortable()
                    ->searchable(),
                TextColumn::make('reason')
                    ->label(__mc('Reason'))
                    ->sortable()
                    ->searchable(),
            ])
            ->recordActions([
                Action::make('Reactivate')
                    ->icon('heroicon-s-check-circle')
                    ->action(fn (Suppression $record) => $this->reactivate($record))
                    ->requiresConfirmation()
                    ->hidden(fn (Suppression $record) => $record->reason === SuppressionReason::spamComplaint)
                    ->label(__mc('Reactivate')),
            ])
            ->headerActions([
                BulkAction::make('Reactivate')
                    ->icon('heroicon-s-check-circle')
                    ->action(fn (Collection $records) => $records->each(fn ($record) => $this->reactivate($record)))
                    ->requiresConfirmation()
                    ->deselectRecordsAfterCompletion()
                    ->authorizeIndividualRecords('reactivate'),
            ])
            ->checkIfRecordIsSelectableUsing(fn (Suppression $record) => Auth::user()->can('reactivate', $record))
            ->defaultSort('created_at', 'desc')
            ->emptyStateHeading(__mc('No suppressed emails'))
            ->emptyStateIcon('heroicon-s-no-symbol')
            ->emptyStateDescription(__mc('Smooth sailing! No suppressed emails found'))
            ->emptyStateActions([
                Action::make('docs')
                    ->label(__mc('Learn more about suppressions'))
                    ->url('https://mailcoach.app/resources/learn-mailcoach/advanced/suppressions')
                    ->openUrlInNewTab()
                    ->link(),
            ]);
    }

    public function getTitle(): string
    {
        return __mc('Suppressions');
    }

    public function getLayout(): string
    {
        return 'mailcoach::app.layouts.settings';
    }

    public function getLayoutData(): array
    {
        return [
            'title' => __mc('Suppressions'),
            'create' => 'suppression',
            'createText' => __mc('Create suppression'),
        ];
    }

    protected function reactivate(Suppression $suppression): void
    {
        $this->authorize('reactivate', $suppression);

        $suppression->delete();

        notify("Reactivated `{$suppression->email}` successfully");
    }
}
