<?php

namespace Spatie\Mailcoach\Livewire\Templates;

use Filament\Actions\Action;
use Filament\Actions\ActionGroup;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Support\HtmlString;
use Spatie\Mailcoach\Domain\Template\Models\Template;
use Spatie\Mailcoach\Livewire\TableComponent;

class TemplatesComponent extends TableComponent
{
    use InteractsWithActions;

    public function table(Table $table): Table
    {
        return parent::table($table)
            ->query(self::getTemplateClass()::query())
            ->recordUrl(function (Template $record) {
                return route('mailcoach.templates.edit', $record);
            })
            ->defaultSort('name')
            ->columns([
                TextColumn::make('name')
                    ->sortable()
                    ->searchable(self::getTemplateClass()::count() > $this->defaultPaginationOptions[0])
                    ->size('base')
                    ->extraAttributes(['class' => 'link']),
                TextColumn::make('created_at')
                    ->size('base')
                    ->extraAttributes([
                        'class' => 'tabular-nums',
                    ])
                    ->date(config('mailcoach.date_format'))
                    ->alignRight()
                    ->sortable(),
            ])
            ->recordActions([
                ActionGroup::make([
                    Action::make('Duplicate')
                        ->action(fn (Template $record) => $this->duplicateTemplate($record))
                        ->icon('heroicon-s-document-duplicate')
                        ->label(__mc('Duplicate')),
                    Action::make('Delete')
                        ->action(function (Template $record) {
                            $record->delete();
                            notify(__mc('Template :template was deleted.', ['template' => $record->name]));
                        })
                        ->requiresConfirmation()
                        ->label(__mc('Delete'))
                        ->modalHeading(fn (Template $record) => __mc('Delete :resource', ['resource' => __mc('template')]))
                        ->modalDescription(fn (Template $record) => new HtmlString(__mc('Are you sure you want to delete :resource<br/><strong>:name</strong>?', [
                            'resource' => __mc('template'),
                            'name' => $record->name,
                        ])))
                        ->icon('heroicon-s-trash')
                        ->color('danger'),
                ]),
            ])
            ->emptyStateIcon('heroicon-s-document-text')
            ->emptyStateDescription(__mc('You have not created any templates yet.'))
            ->emptyStateActions([
                Action::make('learn')
                    ->url('https://mailcoach.app/resources/learn-mailcoach/features/templates')
                    ->label(__mc('Learn more about templates'))
                    ->openUrlInNewTab()
                    ->link(),
            ]);
    }

    public function duplicateTemplate(Template $template)
    {
        $this->authorize('create', self::getTemplateClass());

        $duplicateTemplate = self::getTemplateClass()::create([
            'name' => $template->name.' - '.__mc('copy'),
            'html' => $template->html,
            'structured_html' => $template->structured_html,
        ]);

        notify(__mc('Template :template was created.', ['template' => $duplicateTemplate->name]));

        return redirect()->route('mailcoach.templates.edit', $duplicateTemplate);
    }

    public function getTitle(): string
    {
        return __mc('Templates');
    }

    public function getLayoutData(): array
    {
        return [
            'hideBreadcrumbs' => true,
            'create' => 'template',
            'createText' => __mc('Create template'),
        ];
    }
}
